<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Update { // OAI-260103: Push-update handler with backup + deploy.
	private $settings;

	public function __construct( HW_Settings $settings ) {
		$this->settings = $settings;
	}

	public function register() {
		add_action(
			'rest_api_init',
			function () {
				register_rest_route(
					'hw/v1',
					'/push-update',
					array(
						'methods'             => WP_REST_Server::CREATABLE,
						'callback'            => array( $this, 'handle_push_update' ),
						'permission_callback' => '__return_true', // HMAC enforced in handler.
					)
				);
			}
		);
	}

	public function handle_push_update( WP_REST_Request $request ) {
		$options = $this->settings->get_options();

		if ( empty( $options['allow_push_updates'] ) ) {
			return new WP_Error( 'hw_updates_disabled', 'Push updates are disabled.', array( 'status' => 403 ) );
		}

		if ( empty( $options['site_secret'] ) ) {
			return new WP_Error( 'hw_missing_secret', 'Site secret is not configured.', array( 'status' => 400 ) );
		}

		$raw_body = $request->get_body();
		$provided = $request->get_header( 'x-hw-signature' );
		$expected = hash_hmac( 'sha256', $raw_body, $options['site_secret'] );

		if ( ! $provided || ! hash_equals( $expected, $provided ) ) {
			return new WP_Error( 'hw_bad_signature', 'Signature invalid.', array( 'status' => 401 ) );
		}

		$payload = json_decode( $raw_body, true );
		if ( ! is_array( $payload ) ) {
			return new WP_Error( 'hw_bad_payload', 'Payload must be JSON.', array( 'status' => 400 ) );
		}

		$package_url = isset( $payload['package_url'] ) ? esc_url_raw( $payload['package_url'] ) : '';
		$new_version = isset( $payload['version'] ) ? sanitize_text_field( $payload['version'] ) : '';

		if ( empty( $package_url ) || empty( $new_version ) ) {
			return new WP_Error( 'hw_missing_fields', 'package_url and version are required.', array( 'status' => 400 ) );
		}

		$backup_path = $this->backup_current( $new_version );
		if ( is_wp_error( $backup_path ) ) {
			return $backup_path;
		}

		$result = $this->apply_package( $package_url );
		if ( is_wp_error( $result ) ) {
			$this->log_push_event( 'fail', $new_version, $result->get_error_message() ); // OAI-260103
			return $result;
		}

		$this->log_push_event( 'success', $new_version ); // OAI-260103

		return rest_ensure_response(
			array(
				'success'      => true,
				'new_version'  => $new_version,
				'backup_path'  => $backup_path,
				'installed_at' => current_time( 'mysql' ),
			)
		);
	}

	private function backup_current( $new_version ) {
		$backups_dir = trailingslashit( WP_CONTENT_DIR ) . 'hw-backups';
		if ( ! wp_mkdir_p( $backups_dir ) ) {
			return new WP_Error( 'hw_backup_failed', 'Could not create backups directory.' );
		}

		$timestamp = time();
		$tar_path  = trailingslashit( $backups_dir ) . "hubwright-{$new_version}-{$timestamp}.tar";

		try {
			$phar = new PharData( $tar_path );
			$phar->buildFromDirectory( HW_PATH );
			$phar->compress( Phar::GZ );
			unset( $phar );
			@unlink( $tar_path ); // Keep only .tar.gz
		} catch ( Exception $e ) {
			return new WP_Error( 'hw_backup_failed', 'Backup failed: ' . $e->getMessage() );
		}

		return $tar_path . '.gz';
	}

	private function apply_package( $package_url ) {
		include_once ABSPATH . 'wp-admin/includes/file.php';

		$tmp = download_url( $package_url );
		if ( is_wp_error( $tmp ) ) {
			return new WP_Error( 'hw_download_failed', 'Download failed: ' . $tmp->get_error_message() );
		}

		$tmp_dir = trailingslashit( WP_CONTENT_DIR ) . 'hw-updates/hubwright-' . time();
		wp_mkdir_p( $tmp_dir );

		$extracted_dir = '';

		try {
			$phar = new PharData( $tmp );

			if ( preg_match( '/\.gz$/', $package_url ) ) {
				$tar_path = $tmp_dir . '.tar';
				$phar->decompress( $tar_path );
				$phar = new PharData( $tar_path );
			}

			$phar->extractTo( $tmp_dir, null, true );
			unset( $phar );
		} catch ( Exception $e ) {
			@unlink( $tmp );
			return new WP_Error( 'hw_extract_failed', 'Extract failed: ' . $e->getMessage() );
		}

		@unlink( $tmp );

		// Locate the extracted plugin folder.
		if ( is_dir( $tmp_dir . '/hubwright' ) ) {
			$extracted_dir = $tmp_dir . '/hubwright';
		} else {
			$entries = glob( $tmp_dir . '/*', GLOB_ONLYDIR );
			$extracted_dir = ! empty( $entries ) ? $entries[0] : '';
		}

		if ( empty( $extracted_dir ) || ! is_dir( $extracted_dir ) ) {
			return new WP_Error( 'hw_missing_folder', 'Extracted package did not contain the hubwright folder.' );
		}

		$current_dir   = untrailingslashit( HW_PATH );
		$stale_dir     = $current_dir . '-old-' . time();

		// Move current plugin aside, move new in place.
		if ( ! @rename( $current_dir, $stale_dir ) ) {
			return new WP_Error( 'hw_rename_failed', 'Could not rotate current plugin directory.' );
		}

		if ( ! @rename( $extracted_dir, $current_dir ) ) {
			// Try to roll back.
			@rename( $stale_dir, $current_dir );
			return new WP_Error( 'hw_deploy_failed', 'Could not move new plugin into place.' );
		}

		// Clean up extracted outer folder.
		$this->rrmdir( $tmp_dir );

		return true;
	}

	private function rrmdir( $dir ) {
		if ( ! is_dir( $dir ) ) {
			return;
		}
		$items = scandir( $dir );
		foreach ( $items as $item ) {
			if ( $item === '.' || $item === '..' ) {
				continue;
			}
			$path = $dir . DIRECTORY_SEPARATOR . $item;
			if ( is_dir( $path ) ) {
				$this->rrmdir( $path );
			} else {
				@unlink( $path );
			}
		}
		@rmdir( $dir );
	}

	private function log_push_event( $status, $version, $message = '' ) { // OAI-260103: simple push log.
		$dir = trailingslashit( WP_CONTENT_DIR ) . 'hw-backups';
		if ( ! wp_mkdir_p( $dir ) ) {
			return;
		}
		$line = sprintf(
			'[%s] %s version=%s %s',
			date( 'c' ),
			$status,
			$version,
			$message ? 'msg=' . $message : ''
		);
		@file_put_contents( trailingslashit( $dir ) . 'push.log', $line . "\n", FILE_APPEND );
	}
}
