<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Sync { // OAI-260103: Publish-time sync to hub.
	private $settings;

	public function __construct( HW_Settings $settings ) {
		$this->settings = $settings;
	}

	public function register() {
		add_action( 'transition_post_status', array( $this, 'maybe_sync_post' ), 10, 3 );
	}

	public function maybe_sync_post( $new_status, $old_status, $post ) {
		if ( 'publish' !== $new_status || 'publish' === $old_status ) {
			return;
		}

		$options = $this->settings->get_options();
		if ( empty( $options['enable_sync'] ) || empty( $options['hub_url'] ) ) {
			return;
		}

		$payload = $this->build_payload( $post );
		$this->send_to_hub( $options['hub_url'], $options['site_key'], $options['site_secret'], $payload );
	}

	private function build_payload( $post ) {
		$author_id = $post->post_author;
		$author    = get_userdata( $author_id );
		$topics    = wp_get_post_terms( $post->ID, array( 'category', 'post_tag' ), array( 'fields' => 'slugs' ) );

		return array(
			'posts' => array(
				array(
					'url'          => get_permalink( $post ),
					'title'        => get_the_title( $post ),
					'summary'      => wp_trim_words( wp_strip_all_tags( $post->post_content ), 55 ),
					'published_at' => get_post_time( 'c', true, $post ),
					'topics'       => $topics,
					'author'       => array(
						'slug'        => $author ? $author->user_nicename : '',
						'display_name'=> $author ? $author->display_name : '',
						'avatar_url'  => $author ? get_avatar_url( $author_id ) : '',
						'profile_url' => $author ? get_author_posts_url( $author_id ) : '',
					),
				),
			),
		);
	}

	private function send_to_hub( $hub_url, $site_key, $site_secret, $payload ) {
		$endpoint = trailingslashit( $hub_url ) . 'v1/content/ingest';
		$body     = wp_json_encode( array_merge( array( 'site_id' => $site_key ), $payload ) );

		$response = wp_remote_post(
			$endpoint,
			array(
				'headers' => array(
					'Content-Type'      => 'application/json',
					'X-Site-Key'        => $site_key,
					'X-Site-Signature'  => hash_hmac( 'sha256', $body, $site_secret ),
				),
				'body'    => $body,
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			error_log( '[HW] Hub sync failed: ' . $response->get_error_message() );
		}
	}
}
