<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Settings { // OAI-260103: HubWright settings container.
	private $option_key = 'hubw_settings';

	public function register() {
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	public function add_menu() {
		add_options_page(
			'hubWright',
			'hubWright',
			'manage_options',
			'hw-settings',
			array( $this, 'render_settings_page' )
		);
	}

	public function register_settings() {
		register_setting(
			'hw_settings_group',
			$this->option_key,
			array( $this, 'sanitize' )
		);

		add_settings_section( 'hw_access', 'WordierPress Access', array( $this, 'render_section_intro' ), 'hw-settings' );
		add_settings_section( 'hw_shared', 'Shared Content', array( $this, 'render_section_intro' ), 'hw-settings' );
		add_settings_section( 'hw_authori', 'Authori', array( $this, 'render_section_intro' ), 'hw-settings' );
		add_settings_section( 'hw_paywall', 'Paywall', array( $this, 'render_section_intro' ), 'hw-settings' );
		add_settings_section( 'hw_tiers', 'Tiers', array( $this, 'render_section_intro' ), 'hw-settings' );

		add_settings_field(
			'site_key',
			'WordierPress Email',
			array( $this, 'render_input' ),
			'hw-settings',
			'hw_access',
			array(
				'id'    => 'site_key',
				'label' => 'Email you use on wordierpress.com.',
			)
		);

		add_settings_field(
			'site_secret',
			'Shared Secret',
			array( $this, 'render_input' ),
			'hw-settings',
			'hw_access',
			array(
				'id'    => 'site_secret',
				'label' => 'Generate on https://wordierpress.com/gen_secret; keep private.',
				'type'  => 'password',
				'note'  => '<a href="https://wordierpress.com/get_shared_secret" target="_blank" rel="noreferrer">Get your shared secret</a>',
			)
		);

		add_settings_field(
			'hub_url',
			'Hub URL',
			array( $this, 'render_input' ),
			'hw-settings',
			'hw_access',
			array(
				'id'    => 'hub_url',
				'label' => 'https://hub.example.com',
			)
		);

		add_settings_field(
			'enable_sync',
			'Send posts to hub',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_shared',
			array(
				'id' => 'enable_sync',
			)
		);

		add_settings_field(
			'allow_push_updates',
			'Allow hub push updates',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_shared',
			array(
				'id' => 'allow_push_updates', // OAI-260103: Enable remote update ingestion.
			)
		);

		add_settings_field(
			'enable_paywall',
			'Enable paywall gating',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_paywall',
			array(
				'id' => 'enable_paywall',
				'label_text' => 'Require login to view gated content.',
			)
		);

		add_settings_field(
			'paywall_show_excerpt',
			'Show excerpt when gated',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_paywall',
			array(
				'id' => 'paywall_show_excerpt',
				'label_text' => 'Show the post excerpt above the paywall notice.',
			)
		);

		add_settings_field(
			'paywall_message',
			'Paywall message',
			array( $this, 'render_textarea' ),
			'hw-settings',
			'hw_paywall',
			array(
				'id'    => 'paywall_message',
				'label' => 'Message shown when a post is gated.',
				'rows'  => 3,
			)
		);

		add_settings_field(
			'enable_authori',
			'Enable Authori stamping',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_authori',
			array(
				'id'         => 'enable_authori',
				'label_text' => 'Allow posts to be stamped to the Authori ledger.',
			)
		);

		add_settings_field(
			'authori_auto',
			'Auto-stamp on publish',
			array( $this, 'render_checkbox' ),
			'hw-settings',
			'hw_authori',
			array(
				'id'         => 'authori_auto',
				'label_text' => 'Stamp posts to Authori when they are published.',
			)
		);

		add_settings_field(
			'tiers',
			'Define tiers (Free + up to 5 paid)',
			array( $this, 'render_tiers_field' ),
			'hw-settings',
			'hw_tiers',
			array()
		);
	}

	public function render_settings_page() {
		?>
		<div class="wrap" style="max-width:1200px;">
			<h1>hubWright</h1>
			<form method="post" action="options.php" style="background:#fff;border:1px solid #e5e7eb;padding:20px 24px;border-radius:6px;">
				<?php
				settings_fields( 'hw_settings_group' );
				do_settings_sections( 'hw-settings' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	public function render_input( $args ) {
		$options = $this->get_options();
		$id      = esc_attr( $args['id'] );
		$type    = isset( $args['type'] ) ? esc_attr( $args['type'] ) : 'text';
		$value   = isset( $options[ $id ] ) ? esc_attr( $options[ $id ] ) : '';
		?>
		<input type="<?php echo $type; ?>" class="regular-text" name="<?php echo esc_attr( $this->option_key ); ?>[<?php echo $id; ?>]" id="<?php echo $id; ?>" value="<?php echo $value; ?>" placeholder="<?php echo esc_attr( $args['label'] ); ?>" />
		<?php if ( ! empty( $args['note'] ) ) : ?>
			<p class="description"><?php echo $args['note']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></p>
		<?php endif; ?>
		<?php
	}

	public function render_section_intro() {
		echo '<p style="margin:6px 0 12px 0;color:#475569;">Configure the items in this group.</p>';
	}

	public function render_checkbox( $args ) {
		$options = $this->get_options();
		$id      = esc_attr( $args['id'] );
		$checked = ! empty( $options[ $id ] ) ? 'checked' : '';
		?>
		<label for="<?php echo $id; ?>">
			<input type="checkbox" name="<?php echo esc_attr( $this->option_key ); ?>[<?php echo $id; ?>]" id="<?php echo $id; ?>" value="1" <?php echo $checked; ?> />
			<?php echo isset( $args['label_text'] ) ? esc_html( $args['label_text'] ) : 'Enable content sync to the hub'; ?>
		</label>
		<?php
	}

	public function render_textarea( $args ) {
		$options = $this->get_options();
		$id      = esc_attr( $args['id'] );
		$value   = isset( $options[ $id ] ) ? esc_textarea( $options[ $id ] ) : '';
		$rows    = isset( $args['rows'] ) ? intval( $args['rows'] ) : 3;
		?>
		<textarea class="large-text" name="<?php echo esc_attr( $this->option_key ); ?>[<?php echo $id; ?>]" id="<?php echo $id; ?>" rows="<?php echo $rows; ?>" placeholder="<?php echo esc_attr( $args['label'] ); ?>"><?php echo $value; ?></textarea>
		<?php
	}

	public function render_tiers_field() {
		$options = $this->get_options();
		$tiers   = isset( $options['tiers'] ) && is_array( $options['tiers'] ) ? $options['tiers'] : array();
		$paid    = array_slice( $tiers, 1 );
		$paid    = array_values( $paid );
		for ( $i = 0; $i < 5; $i++ ) {
			if ( ! isset( $paid[ $i ] ) ) {
				$paid[ $i ] = array( 'name' => '', 'desc' => '', 'price' => '' );
			}
		}
		?>
		<table class="form-table">
			<tr>
				<th scope="row">1. Free</th>
				<td>
					<strong>Free</strong><br>
					<label>Description</label><br>
					<input type="text" name="<?php echo esc_attr( $this->option_key ); ?>[tiers_free_desc]" value="<?php echo esc_attr( isset( $tiers[0]['desc'] ) ? $tiers[0]['desc'] : '' ); ?>" class="regular-text" />
					<br><label>Price / month</label><br>
					<input type="text" name="<?php echo esc_attr( $this->option_key ); ?>[tiers_free_price]" value="<?php echo esc_attr( isset( $tiers[0]['price'] ) ? $tiers[0]['price'] : '' ); ?>" class="regular-text" />
				</td>
			</tr>
			<?php for ( $i = 1; $i <= 5; $i++ ) : ?>
				<?php $tier = $paid[ $i - 1 ]; ?>
				<tr>
					<th scope="row"><?php echo intval( $i + 1 ); ?>.</th>
					<td>
						<label>Name</label><br>
						<input type="text" name="<?php echo esc_attr( $this->option_key ); ?>[tiers_paid][<?php echo intval( $i ); ?>][name]" value="<?php echo esc_attr( $tier['name'] ); ?>" class="regular-text" /><br>
						<label>Description</label><br>
						<input type="text" name="<?php echo esc_attr( $this->option_key ); ?>[tiers_paid][<?php echo intval( $i ); ?>][desc]" value="<?php echo esc_attr( $tier['desc'] ); ?>" class="regular-text" /><br>
						<label>Price / month</label><br>
						<input type="text" name="<?php echo esc_attr( $this->option_key ); ?>[tiers_paid][<?php echo intval( $i ); ?>][price]" value="<?php echo esc_attr( $tier['price'] ); ?>" class="regular-text" />
					</td>
				</tr>
			<?php endfor; ?>
		</table>
		<p class="description">Free is mandatory. Up to 5 paid tiers appear in the Posts table dropdown.</p>
		<?php
	}

	public function sanitize( $input ) {
		// Start from existing options to avoid wiping unrelated settings on partial saves. // OAI-260104
		$clean = $this->get_options();
		if ( isset( $input['hub_url'] ) ) {
			$clean['hub_url'] = esc_url_raw( trim( $input['hub_url'] ) );
		}
		if ( isset( $input['site_key'] ) ) {
			$clean['site_key'] = sanitize_text_field( $input['site_key'] );
		}
		if ( isset( $input['site_secret'] ) ) {
			$clean['site_secret'] = sanitize_text_field( $input['site_secret'] );
		}
		$clean['enable_sync'] = ! empty( $input['enable_sync'] ) ? 1 : 0;
		$clean['allow_push_updates'] = ! empty( $input['allow_push_updates'] ) ? 1 : 0;
		$clean['enable_paywall'] = ! empty( $input['enable_paywall'] ) ? 1 : 0;
		$clean['paywall_show_excerpt'] = ! empty( $input['paywall_show_excerpt'] ) ? 1 : 0;
		if ( isset( $input['paywall_message'] ) ) {
			$clean['paywall_message'] = wp_kses_post( $input['paywall_message'] );
		}
		$clean['enable_authori'] = ! empty( $input['enable_authori'] ) ? 1 : 0;
		$clean['authori_auto']   = ! empty( $input['authori_auto'] ) ? 1 : 0;
		// Tiers: Free + up to 5 paid.
		$tiers = array();
		$free_desc  = isset( $input['tiers_free_desc'] ) ? sanitize_text_field( $input['tiers_free_desc'] ) : '';
		$free_price = isset( $input['tiers_free_price'] ) ? sanitize_text_field( $input['tiers_free_price'] ) : '';
		$tiers[] = array(
			'name'  => 'Free',
			'desc'  => $free_desc,
			'price' => $free_price,
		);
		if ( isset( $input['tiers_paid'] ) && is_array( $input['tiers_paid'] ) ) {
			foreach ( $input['tiers_paid'] as $row ) {
				$name = isset( $row['name'] ) ? trim( (string) $row['name'] ) : '';
				if ( $name === '' ) {
					continue;
				}
				$tiers[] = array(
					'name'  => sanitize_text_field( $name ),
					'desc'  => isset( $row['desc'] ) ? sanitize_text_field( $row['desc'] ) : '',
					'price' => isset( $row['price'] ) ? sanitize_text_field( $row['price'] ) : '',
				);
			}
		}
		$clean['tiers'] = $tiers;

		return $clean;
	}

	public function get_options() {
		$defaults = array(
			'hub_url'    => '',
			'site_key'   => '',
			'site_secret'=> '',
			'enable_sync'=> 0,
			'allow_push_updates' => 0,
			'enable_paywall' => 0,
			'paywall_show_excerpt' => 0,
			'paywall_message' => 'Full content is available via WordierPress. Please sign-in for access.', // OAI-260104: default paywall message.
			'enable_authori' => 0,
			'authori_auto'   => 0,
			'tiers'          => array(
				array(
					'name'  => 'Free',
					'desc'  => '',
					'price' => '',
				),
			),
		);

		$options = get_option( $this->option_key, array() );
		if ( empty( $options ) ) {
			$legacy = get_option( 'hw_settings', array() ); // OAI-260104: migrate old key to hubw_ prefix.
			if ( ! empty( $legacy ) ) {
				update_option( $this->option_key, $legacy );
				$options = $legacy;
			}
		}

		return wp_parse_args( $options, $defaults );
	}

	public function get_tier_names( $options = null ) {
		if ( null === $options ) {
			$options = $this->get_options();
		}
		$tiers = isset( $options['tiers'] ) && is_array( $options['tiers'] ) ? $options['tiers'] : array();
		$names = array();
		foreach ( $tiers as $tier ) {
			if ( is_array( $tier ) && isset( $tier['name'] ) ) {
				$names[] = $tier['name'];
			} elseif ( is_string( $tier ) ) {
				$names[] = $tier;
			}
		}
		if ( empty( $names ) ) {
			$names = array( 'Free' );
		}
		return $names;
	}
}
