<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class HW_Authori { // OAI-260103: Proof-of-authorship stamping (placeholder ledger hook).
	private $settings;
	private $log_file;
	private $hash_meta      = '_hubw_authori_hash';
	private $stamped_meta   = '_hubw_authori_stamped_at';
	private $queue_meta     = 'hubw_authori_queue'; // OAI-260104: editor checkbox flag.

	public function __construct( HW_Settings $settings ) {
		$this->settings = $settings;
		$this->log_file = trailingslashit( WP_CONTENT_DIR ) . 'hw-authori.log';
	}

	public function register() {
		register_post_meta(
			'post',
			$this->queue_meta,
			array(
				'type'         => 'boolean',
				'single'       => true,
				'show_in_rest' => true,
				'auth_callback' => function() {
					return current_user_can( 'edit_posts' );
				},
			)
		);
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ) );
		add_action( 'save_post', array( $this, 'maybe_autostamp_on_save' ), 20, 2 );
		add_action( 'admin_post_hw_stamp_authori', array( $this, 'handle_manual_stamp' ) );
	}

	public function add_meta_box() {
		add_meta_box(
			'hw-authori',
			'Authori (proof of authorship)',
			array( $this, 'render_meta_box' ),
			'post',
			'side',
			'low'
		);
	}

	public function render_meta_box( $post ) {
		wp_nonce_field( 'hw_authori_nonce', 'hw_authori_nonce_field' );
		$hash     = get_post_meta( $post->ID, $this->hash_meta, true );
		$stamped  = get_post_meta( $post->ID, $this->stamped_meta, true );
		if ( '' === $hash ) {
			$hash = get_post_meta( $post->ID, '_hw_authori_hash', true ); // OAI-260104: legacy meta fallback.
		}
		if ( '' === $stamped ) {
			$stamped = get_post_meta( $post->ID, '_hw_authori_stamped_at', true ); // OAI-260104: legacy meta fallback.
		}
		$enabled  = ! empty( $this->settings->get_options()['enable_authori'] );
		$auto     = ! empty( $this->settings->get_options()['authori_auto'] );
		?>
		<p>Authori stamps a hash of this post to an append-only ledger for immutable authorship proof.</p>
		<?php if ( $stamped ) : ?>
			<p><strong>Last stamped:</strong> <?php echo esc_html( $stamped ); ?></p>
		<?php endif; ?>
		<?php if ( $hash ) : ?>
			<p style="word-wrap:break-word;"><strong>Hash:</strong> <code><?php echo esc_html( $hash ); ?></code></p>
		<?php endif; ?>
		<p><strong>Global status:</strong> <?php echo $enabled ? 'Enabled' : 'Disabled'; ?><?php echo $auto ? ' (auto on publish)' : ''; ?></p>
		<?php if ( ! $enabled ) : ?>
			<p style="color:#b91c1c;">Enable Authori in hubWright settings to stamp.</p>
		<?php endif; ?>
		<p>
			<button class="button" type="submit" form="post" name="hw_authori_manual" value="1" <?php disabled( ! $enabled ); ?>>Stamp to Authori</button>
		</p>
		<?php
	}

	public function maybe_autostamp_on_save( $post_id, $post ) {
		// Only handle published posts; avoid autosave/revision.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( wp_is_post_revision( $post_id ) ) {
			return;
		}
		if ( 'post' !== $post->post_type ) {
			return;
		}
		$options = $this->settings->get_options();
		if ( empty( $options['enable_authori'] ) ) {
			return;
		}
		$should_queue = get_post_meta( $post_id, $this->queue_meta, true );
		if ( isset( $_POST[ $this->queue_meta ] ) ) {
			$should_queue = (bool) $_POST[ $this->queue_meta ];
		}
		if ( $should_queue ) {
			$this->stamp_post( $post_id, 'editor-checkbox' );
			return;
		}
		// Manual stamp via meta box button.
		if ( isset( $_POST['hw_authori_manual'], $_POST['hw_authori_nonce_field'] ) && wp_verify_nonce( $_POST['hw_authori_nonce_field'], 'hw_authori_nonce' ) ) {
			$this->stamp_post( $post_id, 'manual' );
			return;
		}
		// Auto stamp on publish transition.
		if ( ! empty( $options['authori_auto'] ) && 'publish' === $post->post_status ) {
			$this->stamp_post( $post_id, 'auto' );
		}
	}

	public function handle_manual_stamp() {
		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_die( 'Unauthorized' );
		}
		$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
		if ( ! $post_id || ! isset( $_POST['hw_authori_nonce_field'] ) || ! wp_verify_nonce( $_POST['hw_authori_nonce_field'], 'hw_authori_nonce' ) ) {
			wp_die( 'Invalid request' );
		}
		$this->stamp_post( $post_id, 'manual' );
		$redirect = add_query_arg(
			array(
				'hw_authori' => 'stamped',
				'hw_msg'     => rawurlencode( 'Stamped to Authori.' ),
			),
			admin_url( 'post.php?post=' . $post_id . '&action=edit' )
		);
		wp_safe_redirect( $redirect );
		exit;
	}

	private function stamp_post( $post_id, $mode = 'manual' ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return;
		}
		// Build hash payload.
		$payload = array(
			'post_id'    => $post_id,
			'title'      => get_the_title( $post ),
			'url'        => get_permalink( $post ),
			'author_id'  => (int) $post->post_author,
			'paywalled'  => get_post_meta( $post_id, '_hw_paywall_gate', true ) === '1',
			'mode'       => $mode,
			'timestamp'  => gmdate( 'c' ),
		);
		$hash_source = wp_json_encode( $payload ) . $post->post_content;
		$hash        = hash( 'sha256', $hash_source );
		$payload['hash'] = $hash;

		// Persist hash metadata on the post.
		update_post_meta( $post_id, $this->hash_meta, $hash );
		update_post_meta( $post_id, $this->stamped_meta, $payload['timestamp'] );
		delete_post_meta( $post_id, '_hw_authori_hash' );
		delete_post_meta( $post_id, '_hw_authori_stamped_at' );

		// Append log entry (JSONL) for visibility. // OAI-260103
		$line = wp_json_encode( $payload ) . PHP_EOL;
		@file_put_contents( $this->log_file, $line, FILE_APPEND | LOCK_EX );
	}

	// Editor checkbox now merged into shared WordierPress panel in paywall assets.

	// OAI-260103: expose recent Authori stamps for admin UI.
	public function list_log_entries( $limit = 20 ) {
		if ( ! file_exists( $this->log_file ) ) {
			return array();
		}
		$lines = file( $this->log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );
		$lines = array_slice( array_reverse( $lines ), 0, $limit );
		$entries = array();
		foreach ( $lines as $line ) {
			$decoded = json_decode( $line, true );
			if ( is_array( $decoded ) ) {
				$entries[] = $decoded;
			}
		}
		return $entries;
	}
}
